/*
XHTML WYSIWYG editor, a replacement for textarea's
copyright (c) 2010 by Royal Shitware Inc (RSI) / Rob Thomassen (info@shitware.nl)

This program is free software: you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License
as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License along with this program. If not, see
<http://www.gnu.org/licenses/>.

Version history:
- 2010-04-11/0.1: Go!
- 2010-05-06/0.2: First release.

Requires prototype.js (http://www.prototypejs.org)

Based on:
- openWYSIWYG v1.47 Copyright (c) 2006 openWebWare.com
- TinyEditor (http://www.leigeber.com/2010/02/javascript-wysiwyg-editor)

See also:
- https://developer.mozilla.org/en/Rich-Text_Editing_in_Mozilla
- http://msdn.microsoft.com/en-us/library/aa220275%28office.11%29.aspx (rich-text editing in IE)
*/

var xMemoLangStr = {
  //buttons
  backcolor:'Back color',
  bold:'Bold',
  copy:'Copy',
  create_link:'Insert hyperlink',
  cut:'Cut',
  fontname:'Select font',
  fontsize:'Select size',
  forecolor:'Fore color',
  formatblock:'Select heading',
  help:'Help',
  indent:'Indent',
  insert_image:'Insert image',
  insert_ordered_list:'Ordered list',
  insert_table:'Insert table',
  insert_unordered_list:'Unordered list',
  italic:'Italic',
  justify_center:'Align center',
  justify_full:'Align full',
  justify_left:'Align left',
  justify_right:'Align right',
  maximize:'Maximize the body',
  outdent:'Outdent',
  paste:'Paste',
  print:'Print',
  redo:'Redo',
  remove_format:'Remove formatting',
  strikethrough:'Strikethrough',
  subscript:'Subscript',
  superscript:'Superscript',
  underline:'Underline',
  undo:'Undo',
  view_source:'View source',
  //dialogs
  button_ok:'Ok',
  button_cancel:'Cancel',
  create_link_title:'Create link',
  insert_image_title:'Insert image',
  insert_table_title:'Insert table',
  help_title:'Help',
  link_url:'URL',
  link_blank:'open in a new window',
  image_url:'URL',
  image_alt:'alternate text',
  image_width:'width',
  image_height:'height',
  table_cols:'columns',
  table_rows:'rows',
  help:'',
  //errors
  error_unsupported:'Your browser does not support or allow this function.'
}

function xMemoConfig(){
  this.toolbars = [
    ['fontname','fontsize','formatblock','bold','italic','underline','strikethrough','forecolor','backcolor','|','subscript','superscript','|','justify_left','justify_center','justify_right','justify_full'],
    ['cut','copy','paste','remove_format','|','undo','redo','|','create_link','insert_image','insert_table','|','insert_unordered_list','insert_ordered_list','outdent','indent','|','view_source','maximize','|','print','help']
  ];
  this.xml = true; //XHTML
  this.cssSrc = undefined;
  this.css = undefined;
  this.buttonX = {
    fontname:620,fontsize:710,formatblock:760,bold:0,italic:20,underline:40,strikethrough:60,forecolor:80,backcolor:100,subscript:120,superscript:140,justify_left:160,justify_center:180,justify_right:200,justify_full:220,
    cut:240,copy:260,paste:280,remove_format:300,undo:320,redo:340,create_link:360,insert_image:380,insert_table:400,insert_unordered_list:420,insert_ordered_list:440,outdent:460,indent:480,view_source:500,maximize:540,print:560,help:580
  }
  this.hoverY = 20;
  this.fonts = [
    {family:'arial,helvetica,sans-serif',descr:'Sans Serif'},
    {family:'times new roman,serif',descr:'Serif'},
    {family:'arial black,sans-serif',descr:'Wide'},
    {family:'arial narrow,sans-serif',descr:'Narrow'},
    {family:'comic sans ms,sans-serif',descr:'Comic Sans MS'},
    {family:'courier new,monospace',descr:'Courier New'},
    {family:'garamond,serif',descr:'Garamond'},
    {family:'georgia,serif',descr:'Georgia'},
    {family:'tahoma,sans-serif',descr:'Tahoma'},
    {family:'trebuchet ms,sans-serif',descr:'Trebuchet MS'},
    {family:'verdana,sans-serif',descr:'Verdana'}
  ];
  this.sizes = [
    {size:1,descr:'Extra small'},
    {size:2,descr:'Small'},
    {size:3,descr:'Medium'},
    {size:4,descr:'Large'},
    {size:5,descr:'XL'},
    {size:6,descr:'XXL'},
    {size:7,descr:'XXX'}
  ];
  this.formats = [
    {tag:'h1',descr:'H1'},
    {tag:'h2',descr:'H2'},
    {tag:'h3',descr:'H3'},
    {tag:'h4',descr:'H4'},
    {tag:'h5',descr:'H5'},
    {tag:'h6',descr:'H6'}
  ]
  this.colorFactorStep = 32;
  this.colorIndexStep = 0.2;
}

function xMemo(textarea,config){

  //check params
  this.textarea = $(textarea);
  if(!this.textarea) throw 'xMemo: textarea "' + textarea + '" does not exist';

  this.config = (typeof(config) == 'object' ? config : new xMemoConfig());

  //internal vars
  this.toolbars = new Array();
  this.selects = new Hash();
  this.recentColorDivs = new Hash();
  this.recentColors = new Hash();

  //functions
  this.getSelection = function(){
    if(this.editorFrame.contentWindow.getSelection) return this.editorFrame.contentWindow.getSelection();
    if(this.editor.getSelection) return this.editor.getSelection();
    return this.editor.selection;
  }

  this.getRange = function(selection){
    return (selection.createRange ? selection.createRange() : selection.getRangeAt(0));
  }

  this.getTextRange = function(element){
    var range = element.parentTextEdit.createTextRange();
    range.moveToElementText(element);
    return range;
  }

  this.findParentNode = function(node,parentTagName){
    parentTagName = parentTagName.toUpperCase();
    while(node && (node.tagName != parentTagName)) node = node.parentNode;
    return node;
  }

  this.findParentFromRange = function(range,parentTagName){
    parentTagName = parentTagName.toUpperCase();
    try{
      if(Prototype.Browser.IE){
        var element = (range.length  ? range.item(0) : range.parentElement());
        element = this.findParentNode(element,parentTagName);
        if(element) return element;

        var rangeUp = range.duplicate();
        rangeUp.collapse(true);
        rangeUp.moveEnd('character',1);
        if(rangeUp.text.length){
          while(rangeUp.compareEndPoints('EndToEnd',range) < 0){
            rangeUp.move('Character');
            return this.findParentFromRange(parentTagName,rangeUp);
          }
        }
      }
      else{
        var node = range.startContainer;
        if(node.nodeType != 3) node = node.childNodes[range.startOffset];
        return this.findParentNode(node,parentTagName);
      }
    }
    catch(e){}
    return null;
  }

  this.insertNodeAtSelection = function(insertNode){
    var selection = this.getSelection();
    var range = selection.getRangeAt(0);
    selection.removeAllRanges();
    range.deleteContents();
    var container = range.startContainer;
    var pos = range.startOffset;
    range = this.editor.createRange();

    if(container.nodeType == 3 && insertNode.nodeType == 3){
      container.insertData(pos,insertNode.data);
      range.setEnd(container,pos + insertNode.length);
      range.setStart(container,pos + insertNode.length);
    }
    else {
      var afterNode,beforeNode;
      if(container.nodeType == 3){
        var textNode = container;
        container = textNode.parentNode;
        var text = textNode.nodeValue;
        beforeNode = document.createTextNode(text.substr(0,pos));
        afterNode = document.createTextNode(text.substr(pos));
        container.insertBefore(afterNode,textNode);
        container.insertBefore(insertNode,afterNode);
        container.insertBefore(beforeNode,insertNode);
        container.removeChild(textNode);
      }
      else{
        afterNode = container.childNodes[pos];
        container.insertBefore(insertNode,afterNode);
      }
      try {
        range.setEnd(afterNode,0);
        range.setStart(afterNode,0);
      }
      catch(e){}
    }
    selection.addRange(range);
  }

  this.getColor = function(command){
    if(Prototype.Browser.Gecko) command = 'HiliteColor';
    var color = this.editor.queryCommandValue(command) + '';
    if(color.substr(0,3) == 'rgb'){ //rgb(0,0,0)
      var components = color.match(/\d+/g);
      color = '';
      components.each(function(component){
        component = parseInt(component,10).toString(16);
        color += '00'.substr(component.length) + component;
      });
    }
    else if(color.match(/^\d+$/)){ //decimal
      color = parseInt(color,10).toString(16);
      color = '000000'.substr(color.length) + color;
    }
    else color = '000000';
    return '#' + color;
  }

  this.addToolbar = function(){
    var toolbar = new Element('div',{'class':'toolbar'});
    this.body.appendChild(toolbar);
    toolbar.appendChild(new Element('button',{'class':'grip'}));
    this.toolbars[this.toolbars.length] = toolbar;
    return toolbar;
  }

  this.addButton = function(buttonId,toolbar){
    var title = xMemoLangStr[buttonId];
    var button = new Element('button',{'class':buttonId,title:title || '',unselectable:'on'});
    toolbar.appendChild(button);
    button.observe('click',this.action.bindAsEventListener(this,buttonId));
    var buttonX = this.config.buttonX[buttonId];
    if(buttonX !== undefined){
      button.style.backgroundPosition = -buttonX + 'px 0';
      button.observe('mouseover',function(event,x,y){
        this.style.backgroundPosition = -x + 'px -' + y + 'px';
      }.bindAsEventListener(button,buttonX,this.config.hoverY));
      button.observe('mouseout',function(event,x){
        this.style.backgroundPosition = -x + 'px 0';
      }.bindAsEventListener(button,buttonX));
    }
    return button;
  }

  this.addSelect = function(selectId,button){
    var select = new Element('div',{'class':'select ' + selectId,style:'position:absolute'});
    this.body.appendChild(select);
    var pos = button.positionedOffset();
    select.style.left = pos.left + 'px';
    select.style.top = (pos.top + button.getHeight()) + 'px';
    select.style.maxHeight = (this.body.getHeight() - select.offsetTop - 10) + 'px';
    select.observe('mousemove',function(){
      if(this.closeSelectId) window.clearTimeout(this.closeSelectId);
    }.bind(this));
    select.observe('mouseout',function(select){
      if(this.closeSelectId) window.clearTimeout(this.closeSelectId);
      this.closeSelectId = Element.hide.delay(0.5,select);
    }.bind(this,select));
    this.selects.set(selectId,select);
    return select;
  }

  this.addRecentColor = function(selectId,color){
    if(this.recentColors.get(selectId).indexOf('|' + color + '|') < 0){
      var button = new Element('button',{style:'background:' + color});
      button.recentColor = color;
      this.recentColorDivs.get(selectId).appendChild(button);
      button.observe('click',this.action.bindAsEventListener(this,selectId,color));
      this.recentColors.set(selectId,this.recentColors.get(selectId) + color + '|');
    }
  }

  this.addColorSelect = function(selectId,button){
    var select = this.addSelect(selectId,button);
    this.recentColorDivs.set(selectId,new Element('div',{'class':'recent'}));
    this.recentColors.set(selectId,'|');
    select.appendChild(this.recentColorDivs.get(selectId));
    this.addRecentColor(selectId,this.getColor(selectId));
    var table = new Element('table');
    select.appendChild(table);
    var r,g,b,x;
    for(f = -256; f <= 256; f += this.config.colorFactorStep){
      var row = table.insertRow(-1);
      for(i = 0; i <= 6.1; i += this.config.colorIndexStep){
        if(i < 1){
          r = 1;
          g = i;
          b = 0;
        }
        else if(i < 2){
          r = 2 - i;
          g = 1;
          b = 0;
        }
        else if(i < 3){
          r = 0;
          g = 1;
          b = i - 2;
        }
        else if(i < 4){
          r = 0;
          g = 4 - i;
          b = 1;
        }
        else if(i < 5){
          r = i - 4;
          g = 0;
          b = 1;
        }
        else if(i < 6){
          r = 1;
          g = 0;
          b = 6 - i;
        }
        else{
          r = 0.5;
          g = 0.5;
          b = 0.5;
        }
        if(f < 0){
          x = Math.max(f,-255);
          r = Math.round(r * 255 - (1 - r) * x);
          g = Math.round(g * 255 - (1 - g) * x);
          b = Math.round(b * 255 - (1 - b) * x);
        }
        else{
          x = Math.max(255 - f,0)
          r = Math.round(r * x);
          g = Math.round(g * x);
          b = Math.round(b * x);
        }
        color = ((r << 16) + (g << 8) + b).toString(16);
        color = '#' + '000000'.substr(color.length) + color;
        var cell = new Element('td',{style:'background:' + color,title:r + ',' + g + ',' + b});
        row.appendChild(cell);
        cell.observe('click',this.action.bindAsEventListener(this,selectId,color));
      }
    }
    return select;
  }

  this.closeSelects = function(exceptId){
    this.selects.each(function(pair){ if(pair.key != exceptId) pair.value.hide() });
  }

  this.createDialog = function(dialogId){
    var dialog = new Element('div',{'class':'xMemo dialog ' + dialogId});
    var pos = this.body.positionedOffset();
    var dim = this.body.getDimensions();
    dialog.style.left = Math.round(pos.left + dim.width / 2) + 'px';
    dialog.style.top = Math.round(pos.top + dim.height / 2) + 'px';
    this.body.parentNode.insertBefore(dialog,this.body);
    dialog.observe('click',function(event){ event.stopPropagation() });
    document.observe('click',function(event){ this.hide() }.bindAsEventListener(dialog));
    var title = new Element('div',{'class':'title'}).update(xMemoLangStr[dialogId + '_title']);
    dialog.appendChild(title);
    title.observe('mousedown',function(event,dialog){
      event.stop();
      this.dragging = true;
      var pos = dialog.positionedOffset();
      this.dragX = event.clientX - pos.left;
      this.dragY = event.clientY - pos.top;
    }.bindAsEventListener(this,dialog));
    document.observe('mousemove',function(event,dialog){
      if(this.dragging){
        event.stop();
        dialog.style.left = (event.clientX - this.dragX) + 'px';
        dialog.style.top = (event.clientY - this.dragY) + 'px';
      }
    }.bindAsEventListener(this,dialog));
    document.observe('mouseup',function(event){ this.dragging = false }.bindAsEventListener(this));
    return dialog;
  }

  this.createDialogTable = function(dialog,callback,colspan){
    var table = new Element('table');
    dialog.appendChild(table);
    var row = table.insertRow(0);
    var cell = new Element('td',{'class':'action',colspan:colspan});
    row.appendChild(cell);
    var button = new Element('input',{type:'button','value':xMemoLangStr['button_ok']});
    cell.appendChild(button);
    button.observe('click',callback.bindAsEventListener(this));
    var button = new Element('input',{type:'button','value':xMemoLangStr['button_cancel']});
    cell.appendChild(button);
    button.observe('click',function(){ this.hide() }.bind(dialog));
    return table;
  }

  this.action = function(event,buttonId,value){
    event.stop();
    this.closeSelects(buttonId);
    switch(buttonId){
      case 'fontname':
        if(this.fontSelect) this.fontSelect.toggle();
        else{
          this.fontSelect = this.addSelect(buttonId,event.target);
          this.config.fonts.each(function(font){
            var option = new Element('div',{'class':'option',style:'font-family:' + font.family,unselectable:'on'}).update(font.descr);
            this.fontSelect.appendChild(option);
            option.observe('click',this.action.bindAsEventListener(this,buttonId,font.family));
          }.bind(this));
        }
        break;
      case 'fontsize':
        if(this.fontSize) this.fontSize.toggle();
        else{
          this.fontSize = this.addSelect(buttonId,event.target);
          this.config.sizes.each(function(size){
            var option = new Element('div',{'class':'option'}).update(new Element('font',{size:size.size,unselectable:'on'}).update(size.descr));
            this.fontSize.appendChild(option);
            option.observe('click',this.action.bindAsEventListener(this,buttonId,size.size));
          }.bind(this));
        }
        break;
      case 'formatblock':
        if(this.formats) this.formats.toggle();
        else{
          this.formats = this.addSelect(buttonId,event.target);
          this.config.formats.each(function(format){
            var option = new Element('div',{'class':'option'}).update(new Element(format.tag,{unselectable:'on'}).update(format.descr));
            this.formats.appendChild(option);
            option.observe('click',this.action.bindAsEventListener(this,buttonId,'<' + format.tag + '>'));
          }.bind(this));
        }
        break;
      case 'forecolor':
        if(this.foreColors) this.foreColors.toggle();
        else this.foreColors = this.addColorSelect(buttonId,event.target);
        if(value) this.addRecentColor(buttonId,value);
        break;
      case 'backcolor':
        if(this.backColors) this.backColors.toggle();
        else this.backColors = this.addColorSelect(buttonId,event.target);
        if(value) this.addRecentColor(buttonId,value);
        if(Prototype.Browser.Gecko) buttonId = 'HiliteColor';
        break;
      case 'create_link':
        if(this.linkDialog) this.linkDialog.show();
        else{
          this.linkDialog = this.createDialog(buttonId);
          var table = this.createDialogTable(this.linkDialog,function(event){
            var update = !!this.link;
            if(!update) this.link = this.editor.createElement('a');
            this.link.setAttribute('href',this.linkUrl.value);
            this.link.setAttribute('target',this.linkBlank.checked ? '_blank' : '');
            if(!update){
              if(Prototype.Browser.IE){
                this.range.select();
                this.link.innerHTML = (this.range.htmlText || this.linkUrl.value.replace(/^[^:]+:\/\//,''));
                this.range.pasteHTML(this.link.outerHTML);
              }
              else{
                var node = this.range.startContainer;
                if(node.nodeType != 3) node = node.childNodes[this.range.startOffset];
                if(node && node.tagName && node.innerHTML) this.link.appendChild(node);
                else this.link.innerHTML = (selection.toString() || this.linkUrl.value.replace(/^[^:]+:\/\//,''));
                this.insertNodeAtSelection(this.link);
              }
            }
            this.linkDialog.hide();
          },2);
          var row = table.insertRow(0);
          row.appendChild(new Element('th').update(xMemoLangStr['link_url']));
          this.linkUrl = new Element('input',{'class':'url'});
          row.appendChild(new Element('td').update(this.linkUrl));
          var row = table.insertRow(1);
          row.appendChild(new Element('th'));
          this.linkBlank = new Element('input',{type:'checkbox'});
          var cell = new Element('td').update(this.linkBlank);
          row.appendChild(cell);
          cell.appendChild(new Element('label').update(xMemoLangStr['link_blank']));
        }
        var selection = this.getSelection();
        this.range = this.getRange(selection);
        if(Prototype.Browser.IE && (selection.type == 'Control') && (this.range.length == 1)){
          this.range = this.getTextRange(selection);
          this.range.select();
        }
        this.link = this.findParentFromRange(this.range,'a');
        this.linkUrl.value = (this.link ? this.link.getAttribute('href') : 'http://');
        this.linkBlank.checked = (this.link ? this.link.getAttribute('target') == '_blank' : false);
        break;
      case 'insert_image':
        if(this.imageDialog) this.imageDialog.show();
        else{
          this.imageDialog = this.createDialog(buttonId);
          var table = this.createDialogTable(this.imageDialog,function(event){
            var update = !!this.image;
            if(!update) this.image = this.editor.createElement('img');
            this.image.setAttribute('src',this.imageUrl.value);
            if(this.imageWidth.value) this.image.setAttribute('width',this.imageWidth.value);
            if(this.imageHeight.value) this.image.setAttribute('height',this.imageHeight.value);
            this.image.setAttribute('alt',this.imageAlt.value);
            if(Prototype.Browser.IE) this.range.pasteHTML(this.image.outerHTML);
            else this.insertNodeAtSelection(this.image);
            this.imageDialog.hide();
          },4);
          var row = table.insertRow(0);
          row.appendChild(new Element('th').update(xMemoLangStr['image_url']));
          this.imageUrl = new Element('input',{'class':'url'});
          row.appendChild(new Element('td',{colspan:3}).update(this.imageUrl));
          var row = table.insertRow(1);
          row.appendChild(new Element('th').update(xMemoLangStr['image_width']));
          this.imageWidth = new Element('input',{'class':'dim'});
          row.appendChild(new Element('td').update(this.imageWidth));
          row.appendChild(new Element('th').update(xMemoLangStr['image_height']));
          this.imageHeight = new Element('input',{'class':'dim'});
          row.appendChild(new Element('td').update(this.imageHeight));
          var row = table.insertRow(2);
          row.appendChild(new Element('th').update(xMemoLangStr['image_alt']));
          this.imageAlt = new Element('input',{'class':'alt'});
          row.appendChild(new Element('td',{colspan:3}).update(this.imageAlt));
        }
        var selection = this.getSelection();
        this.range = this.getRange(selection);
        this.image = this.findParentFromRange(this.range,'img');
        this.imageUrl.value = (this.image ? this.image.getAttribute('src') : 'http://');
        this.imageWidth.value = (this.image ? this.image.getAttribute('width') : '');
        this.imageHeight.value = (this.image ? this.image.getAttribute('height') : '');
        this.imageAlt.value = (this.image ? this.image.getAttribute('alt') : '');
        break;
      case 'insert_table':
        if(this.tableDialog) this.tableDialog.show();
        else{
          this.tableDialog = this.createDialog(buttonId);
          var table = this.createDialogTable(this.tableDialog,function(event){
            var table = new Element('table',{width:'100%',border:1});
            for(r = 0; r < this.tableRows.value; r++){
              var row = table.insertRow(row);
              for(c = 0; c < this.tableCols.value; c++) row.appendChild(new Element('td').update(' '));
            }
            if(Prototype.Browser.IE) this.range.pasteHTML(table.outerHTML);
            else this.insertNodeAtSelection(table);
            this.tableDialog.hide();
          },4);
          var row = table.insertRow(0);
          row.appendChild(new Element('th').update(xMemoLangStr['table_cols']));
          this.tableCols = new Element('input',{'class':'dim','value':2});
          row.appendChild(new Element('td').update(this.tableCols));
          row.appendChild(new Element('th').update(xMemoLangStr['table_rows']));
          this.tableRows = new Element('input',{'class':'dim','value':2});
          row.appendChild(new Element('td').update(this.tableRows));
        }
        break;
      case 'view_source':
        this.save();
        this.editorDiv.hide();
        this.toolbars.invoke('hide');
        this.body.addClassName('view_source');
        this.textarea.show();
        this.textButton.show();
        break;
      case 'view_text':
        this.load();
        this.textButton.hide();
        this.textarea.hide();
        this.body.removeClassName('view_source');
        this.toolbars.invoke('show');
        this.editorDiv.show();
        break;
      case 'maximize':
        this.body.toggleClassName('maximize');
        this.fitEditor();
        break;
      case 'print':
        this.editorFrame.contentWindow.print();
        break;
      case 'help':
        if(this.helpDialog) this.helpDialog.show();
        else{
          this.helpDialog = this.createDialog(buttonId);
          var title = new Element('center');
          title.appendChild(new Element('h1').update('xMemo'));
          title.appendChild(new Element('h4').update('XHTML WYSIWYG editor, a replacement for textarea\'s'));
          title.appendChild(new Element('i').update('copyright © 2010 by <a href="http://projects.shitware.nl" target="_blank">Royal Shitware Inc (RSI)</a> / Rob Thomassen'));
          this.helpDialog.appendChild(title);
          this.helpDialog.appendChild(new Element('hr'));
          this.helpDialog.appendChild(document.createTextNode(xMemoLangStr['help']));
          var button = new Element('input',{type:'button','value':xMemoLangStr['button_ok']});
          this.helpDialog.appendChild(new Element('center').update(button));
          button.observe('click',function(){ this.hide() }.bind(this.helpDialog));
        }
        break;
      case 'seperator':
        break;
      default:
        value = null;
    }
    if(value !== undefined) try{
      this.editor.execCommand(buttonId.replace(/_/g,''),0,value);
    }
    catch(e){
      alert(xMemoLangStr['error_unsupported']);
    }
  }

  this.load = function(){
    var html = this.textarea.value;
    if(this.config.xml && !Prototype.Browser.IE){
      html = html.replace(/<strong>(.*)<\/strong>/gi,'<span style="font-weight: bold;">$1</span>');
      html = html.replace(/<em>(.*)<\/em>/gi,'<span style="font-weight: italic;">$1</span>')
    }
    if(this.editor) this.editor.body.innerHTML = html;
    return html;
  }

  this.save = function(){
    var html = this.editor.body.innerHTML;
    html = html.replace(/(<br ?\/?>)(?![\r\n])/gi,'$1\n');
    if(this.config.xml){
      html = html.replace(/<span class="apple-style-span">(.*)<\/span>/gi,'$1');
      html = html.replace(/ class="apple-style-span"/gi,'');
      html = html.replace(/<span style="">/gi,'');
      html = html.replace(/<br>/gi,'<br />');
      html = html.replace(/<br ?\/?>$/gi,'');
      html = html.replace(/^<br ?\/?>/gi,'');
      html = html.replace(/(<img [^>]+[^\/])>/gi,'$1 />');
      html = html.replace(/<b\b[^>]*>(.*?)<\/b[^>]*>/gi,'<strong>$1</strong>');
      html = html.replace(/<i\b[^>]*>(.*?)<\/i[^>]*>/gi,'<em>$1</em>');
      html = html.replace(/<u\b[^>]*>(.*?)<\/u[^>]*>/gi,'<span style="text-decoration:underline">$1</span>');
      html = html.replace(/<(b|strong|em|i|u) style="font-weight: normal;?">(.*)<\/(b|strong|em|i|u)>/gi,'$2');
      html = html.replace(/<(b|strong|em|i|u) style="(.*)">(.*)<\/(b|strong|em|i|u)>/gi,'<span style="$2"><$4>$3</$4></span>');
      html = html.replace(/<span style="font-weight: normal;?">(.*)<\/span>/gi,'$1');
      html = html.replace(/<span style="font-weight: bold;?">(.*)<\/span>/gi,'<strong>$1</strong>');
      html = html.replace(/<span style="font-style: italic;?">(.*)<\/span>/gi,'<em>$1</em>');
      html = html.replace(/<span style="font-weight: bold;?">(.*)<\/span>|<b\b[^>]*>(.*?)<\/b[^>]*>/gi,'<strong>$1</strong>')
    }
    this.textarea.value = html;
    return true;
  }

  this.enable = function(disable){
    this.body.select('button').each(function(button){ button.disabled = disable });
    if(Prototype.Browser.IE) this.editor.body.contentEditable = !disable;
    else this.editor.designMode = (disable ? 'Off' : 'On');
  }

  this.disable = function(){
    this.enable(true);
  }

  this.fitEditor = function(){
    if(Prototype.Browser.IE){ //others do well with CSS rules
      var toolbar = this.toolbars[this.toolbars.length - 1];
      this.editorDiv.style.height = (this.body.clientHeight - toolbar.positionedOffset().top - toolbar.getHeight()) + 'px';
    }
    this.textarea.style.height = this.body.clientHeight + 'px';
  }

  //init
  this.body = new Element('div',{'class':'xMemo'});
  this.textarea.parentNode.insertBefore(this.body,this.textarea);
  var dim = this.textarea.getDimensions();

  this.textButton = new Element('button',{'class':'view_text'});
  this.body.appendChild(this.textButton);
  this.textButton.observe('click',this.action.bindAsEventListener(this,'view_text'));
  this.textButton.hide();

  this.body.appendChild(this.textarea);
  this.textarea.hide();

  this.body.style.width = dim.width + 'px';
  this.body.style.height = dim.height + 'px';

  this.toolbars = new Array();
  this.config.toolbars.each(function(buttonIds){
    var toolbar = this.addToolbar();
    buttonIds.each(function(buttonId){
      if(buttonId == '|') buttonId = 'seperator';
      var button = this.addButton(buttonId,toolbar);
    }.bind(this));
  }.bind(this));

  this.body.observe('click',function(event){ this.closeSelects() }.bindAsEventListener(this));

  this.editorDiv = new Element('div',{'class':'editor'});
  this.body.appendChild(this.editorDiv);
  var toolbar = this.toolbars[this.toolbars.length - 1];
  this.editorDiv.style.top = (toolbar.positionedOffset().top + toolbar.getHeight()) + 'px';
  this.editorFrame = new Element('iframe',{'class':'editor','frameborder':0});
  this.editorDiv.appendChild(this.editorFrame);
  this.fitEditor();

  var html = '<html><head>';
  if(this.config.cssSrc) html += '<link rel="stylesheet" href="' + this.config.cssSrc + '" />';
  if(this.config.css) html += '<style type="text/css">' + this.config.css + '</style>';
  html += '</head><body>' + this.load() + '</body></html>';

  this.editor = this.editorFrame.contentWindow.document;
  this.editor.open();
  this.editor.write(html);
  this.editor.close();
  this.editor.designMode = 'On';
  if(this.config.xml) try{
    this.editor.execCommand('styleWithCSS',0,0);
  }
  catch(e){}

  this.textarea.style.width = this.body.clientWidth + 'px'
  this.textarea.style.height = this.body.clientHeight + 'px';

  var form = this.textarea.parentNode;
  while(form && form.nodeName.toLowerCase() != 'form') form = form.parentNode;
  if(form) Element.observe(form,'submit',this.save.bind(this));
}